/*
 * Decompiled with CFR 0.152.
 */
package io.olvid.engine.secure_io;

import io.olvid.engine.Logger;
import io.olvid.engine.crypto.exceptions.DecryptionException;
import io.olvid.engine.datatypes.EncryptedBytes;
import io.olvid.engine.datatypes.key.symmetric.AuthEncKey;
import io.olvid.engine.datatypes.key.symmetric.MACKey;
import io.olvid.engine.encoder.Encoded;
import io.olvid.engine.secure_io.KeyManagerSingleton;
import io.olvid.engine.secure_io.SecureFile;
import io.olvid.engine.secure_io.SecureFileHeader;
import io.olvid.engine.secure_io.SecureIOHelper;
import io.olvid.engine.secure_io.datatypes.Pair;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.util.Arrays;

public class SecureFileOutputStream
extends OutputStream {
    private final AccessMode accessMode;
    private SecureFileHeader secureFileHeader;
    private SecureFile secureFile;
    private final byte[] blockBuffer;
    private int blockBufferFullness;
    private long lastByteWrittenPosition;
    private RandomAccessFile fileAccessor;

    public SecureFileOutputStream(SecureFile associatedFSFile) throws IOException {
        this(associatedFSFile, AccessMode.APPEND, 0L);
    }

    public SecureFileOutputStream(SecureFile secureFile, AccessMode accessMode, long truncationOffset) throws IOException {
        if (secureFile == null) {
            throw new IOException();
        }
        if (secureFile.isDirectory()) {
            Logger.e("Can't open secure output stream on directory");
            throw new IOException();
        }
        this.lastByteWrittenPosition = 4096L;
        this.accessMode = accessMode;
        this.blockBuffer = new byte[4056];
        this.blockBufferFullness = 0;
        this.secureFile = secureFile;
        this.buildHeader(truncationOffset);
    }

    @Override
    public void write(int b) throws IOException {
        byte[] byteToWrite = new byte[]{(byte)(b & 0xFF)};
        this.write(byteToWrite, 0, 1);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (len == 0) {
            Logger.w("Nothing to write....or inconsistent args?\ngiven length : " + len);
            return;
        }
        try {
            boolean rewriteHeader = false;
            while (len > 0) {
                int bytesToRead = Math.min(len, 4056 - this.blockBufferFullness);
                System.arraycopy(b, off, this.blockBuffer, this.blockBufferFullness, bytesToRead);
                this.blockBufferFullness += bytesToRead;
                off += bytesToRead;
                len -= bytesToRead;
                if (this.blockBufferFullness != 4056) continue;
                EncryptedBytes cipheredBuffer = SecureIOHelper.authEnc.encrypt(this.secureFileHeader.getAssociatedKeys().getFileContentEncryptionKey(), this.blockBuffer, SecureIOHelper.prng);
                this.fileAccessor.seek(this.lastByteWrittenPosition / 4096L * 4096L);
                this.fileAccessor.write(cipheredBuffer.getBytes());
                this.lastByteWrittenPosition = this.lastByteWrittenPosition / 4096L * 4096L + (long)cipheredBuffer.length;
                this.blockBufferFullness = 0;
                rewriteHeader = true;
            }
            if (rewriteHeader) {
                this.updateHeader();
            }
        }
        catch (InvalidKeyException e) {
            Logger.e("Invalid Key : \n" + e.getMessage(), e);
        }
    }

    private void updateHeader() throws IOException {
        long blocks = this.lastByteWrittenPosition / 4096L - 1L;
        long lastBlockSize = this.lastByteWrittenPosition % 4096L;
        long plainTextSize = blocks * 4056L;
        if (lastBlockSize != 0L) {
            plainTextSize += (long)SecureIOHelper.authEnc.plaintextLengthFromCiphertextLength((int)lastBlockSize);
        }
        this.secureFileHeader.setFileSize(plainTextSize);
        byte[] header = this.secureFileHeader.buildHeaderBlock(this.secureFileHeader.getAssociatedKeys().getFileNameEncryptionKeys().getSecond());
        this.fileAccessor.seek(0L);
        this.fileAccessor.write(header);
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.fileAccessor.close();
    }

    @Override
    public void flush() throws IOException {
        try {
            if (this.blockBufferFullness > 0) {
                byte[] lastBlock = Arrays.copyOfRange(this.blockBuffer, 0, this.blockBufferFullness);
                EncryptedBytes toWrite = SecureIOHelper.authEnc.encrypt(this.secureFileHeader.getAssociatedKeys().getFileContentEncryptionKey(), lastBlock, SecureIOHelper.prng);
                this.fileAccessor.seek(this.lastByteWrittenPosition / 4096L * 4096L);
                this.fileAccessor.write(toWrite.getBytes());
                this.lastByteWrittenPosition = this.lastByteWrittenPosition / 4096L * 4096L + (long)toWrite.length;
                this.updateHeader();
            }
        }
        catch (InvalidKeyException e) {
            Logger.e("Invalid Key Exception : \n " + e.getMessage(), e);
            throw new IOException();
        }
    }

    public SecureFileHeader getSecureFileHeader() {
        return this.secureFileHeader;
    }

    private void buildHeader(long truncationOffset) {
        try {
            File dir;
            if (this.secureFile.getFSNameFile() == null) {
                throw new IOException();
            }
            if (this.secureFile.getFSNameFile().getParent() != null && !(dir = new File(this.secureFile.getFSNameFile().getParent())).exists()) {
                throw new FileNotFoundException("Parent directory not found");
            }
            this.fileAccessor = new RandomAccessFile(this.secureFile.getFSNameFile(), "rw");
            if (this.fileAccessor.length() != 0L) {
                this.initHeaderFromFS(truncationOffset, this.secureFile.getFSNameFile().getName());
            } else {
                Pair<MACKey, AuthEncKey> fileNameKeyPair = this.secureFile.getAssociatedKeys().getFileNameEncryptionKeys();
                EncryptedBytes ctrCypheredFileName = SecureIOHelper.authEnc.encrypt(fileNameKeyPair.getSecond(), this.secureFile.getPlainNameFile().getName().getBytes(StandardCharsets.UTF_8), SecureIOHelper.prng);
                AuthEncKey authEncKey = KeyManagerSingleton.getInstance().generateKeyForFileEnc();
                this.secureFile.getAssociatedKeys().setFileContentEncryptionKey(authEncKey);
                this.secureFileHeader = new SecureFileHeader(0L, ctrCypheredFileName.getBytes(), Encoded.of(authEncKey), this.secureFile.getAssociatedKeys(), this.secureFile.getPlainNameFile().getName().getBytes(StandardCharsets.UTF_8));
                byte[] headerBlock = this.secureFileHeader.buildHeaderBlock(this.secureFile.getAssociatedKeys().getFileNameEncryptionKeys().getSecond());
                this.fileAccessor.seek(0L);
                this.fileAccessor.write(headerBlock);
            }
        }
        catch (FileNotFoundException | InvalidKeyException e) {
            Logger.e("Invalid Key Exception : \n" + e.getMessage(), e);
        }
        catch (IOException exception) {
            Logger.e("IO Exception : \n " + exception.getMessage(), exception);
        }
    }

    private void initHeaderFromFS(long truncationOffset, String FSFileName) throws IOException {
        this.secureFileHeader = SecureIOHelper.getSecureFileHeaderFromFS(this.fileAccessor, FSFileName, true);
        if (this.secureFileHeader == null) {
            return;
        }
        switch (this.accessMode.ordinal()) {
            case 1: {
                this.setBufferFullnessAndWritePosition(this.fileAccessor.length());
                break;
            }
            case 0: {
                long encryptedTruncationOffset = SecureIOHelper.computeEncryptedDataOffsetFromPlainDataOffset(truncationOffset);
                this.secureFileHeader.setFileSize(truncationOffset);
                byte[] headerBytes = this.secureFileHeader.buildHeaderBlock(this.secureFileHeader.getAssociatedKeys().getFileNameEncryptionKeys().getSecond());
                this.fileAccessor.seek(0L);
                this.fileAccessor.write(headerBytes);
                this.setBufferFullnessAndWritePosition(encryptedTruncationOffset);
                this.fileAccessor.setLength(encryptedTruncationOffset);
                this.fileAccessor.seek(encryptedTruncationOffset);
                break;
            }
            default: {
                Logger.e("Unknown file access mode");
                throw new IOException();
            }
        }
    }

    private void setBufferFullnessAndWritePosition(long writeOffset) throws IOException {
        byte[] tmpBf = new byte[(int)(writeOffset % 4096L)];
        if (tmpBf.length > 0) {
            this.fileAccessor.seek(writeOffset - (long)tmpBf.length);
            SecureIOHelper.bulletProofRead(this.fileAccessor, tmpBf);
            try {
                byte[] tmpBf1 = SecureIOHelper.authEnc.decrypt(this.secureFileHeader.getAssociatedKeys().getFileContentEncryptionKey(), new EncryptedBytes(tmpBf));
                System.arraycopy(tmpBf1, 0, this.blockBuffer, 0, tmpBf1.length);
                this.blockBufferFullness = tmpBf1.length;
                this.lastByteWrittenPosition = writeOffset;
            }
            catch (InvalidKeyException e) {
                Logger.e("Invalid Key Exception : \n" + e.getMessage(), e);
                throw new IOException();
            }
            catch (DecryptionException e) {
                Logger.e("Decryption error : \n" + e.getMessage(), e);
                throw new IOException();
            }
        }
    }

    public static enum AccessMode {
        TRUNCATE,
        APPEND;

    }
}

